
/**
 * ============================================================
 * AISELF 后端核心入口
 * server.js
 *
 * 职责定位：
 * - 作为前端与 DeepSeek API 之间的唯一安全桥梁
 * - 固化“五 Agent 架构”中的 AgentC（编排执行器）
 * - 约束 AgentE（模型）严格遵守证据优先、人格仅作风格
 *
 * 技术栈：
 * - Node.js + Express
 * - OpenAI SDK（兼容 DeepSeek API）
 * ============================================================
 */

import express from "express";
import cors from "cors";
import dotenv from "dotenv";
import OpenAI from "openai";

// ------------------------------------------------------------
// 1. 基础初始化
// ------------------------------------------------------------

// 读取 .env 文件中的环境变量
dotenv.config();

// 创建 Express 应用
const app = express();

// 允许跨域（开发阶段先全开，上线可收紧）
app.use(cors({
  origin: "https://aiself.offermore.london",  // ✅ 只允许你的前端子域名访问
  methods: ["GET", "POST", "OPTIONS"],
  allowedHeaders: ["Content-Type"]
}));


// 解析 JSON 请求体，限制大小防止滥用
app.use(express.json({ limit: "2mb" }));

// 端口
const PORT = process.env.PORT || 3000;

// ------------------------------------------------------------
// 2. 初始化 DeepSeek 客户端（OpenAI SDK 兼容）
// ------------------------------------------------------------

// ⚠️ API Key 只存在于后端
// ⚠️ baseURL 使用 DeepSeek 文档指定地址
const openai = new OpenAI({
  baseURL: "https://api.deepseek.com",
  apiKey: process.env.DEEPSEEK_API_KEY,
});

// ------------------------------------------------------------
// 3. 健康检查接口（用于确认后端是否启动成功）
// ------------------------------------------------------------

app.get("/health", (req, res) => {
  res.json({
    ok: true,
    service: "aiself-backend",
    time: new Date().toISOString(),
  });
});

// ------------------------------------------------------------
// 4. /api/chat —— 核心接口
//    这是前端 chat1.js 唯一允许调用的 AI 接口
// ------------------------------------------------------------

app.post("/api/chat", async (req, res) => {
  try {
    /**
     * 前端必须传入的结构（强约束）：
     * {
     *   agent: "E",
     *   persona_spec: {...},      // 来自 AgentB
     *   evidence_pack: {...},     // 来自 AgentD
     * }
     */
    const { agent, persona_spec, evidence_pack } = req.body || {};

    // --------------------------------------------------------
    // 4.1 基础校验（AgentC 的第一道防线）
    // --------------------------------------------------------

    if (agent !== "E") {
      return res.status(400).json({
        error: "Only AgentE execution is allowed on this endpoint.",
      });
    }

    if (!persona_spec) {
      return res.status(400).json({
        error: "Missing persona_spec (from AgentB).",
      });
    }

    if (!evidence_pack || !evidence_pack.user_question) {
      return res.status(400).json({
        error: "Missing evidence_pack or user_question (from AgentD).",
      });
    }

    // --------------------------------------------------------
    // 4.2 AgentC：生成 task_brief（融合但不回答）
    // --------------------------------------------------------

    /**
     * AgentC 的核心原则：
     * - persona_spec 只作为「风格与决策规则」
     * - evidence_pack 是「唯一事实来源」
     * - 明确区分：确定 / 不确定 / 推测
     */

    const task_brief = {
      persona_constraints: {
        // 从 persona_spec 中抽取“行为规则”，而不是故事
        identity_summary: persona_spec.identity_summary,
        core_values: persona_spec.core_values,
        thinking_heuristics: persona_spec.thinking_heuristics,
        speech_style: persona_spec.speech_style,
        taboos: persona_spec.taboos,
        uncertainty_policy: persona_spec.uncertainty_policy,
      },

      grounded_facts: {
        user_question: evidence_pack.user_question,
        source_list: evidence_pack.source_list || [],
        key_excerpts: evidence_pack.key_excerpts || [],
        facts: evidence_pack.facts || [],
      },

      hypotheses: [], // 明确标注为“推测”（如果有）

      answer_plan: [
        "首先基于证据回答可以确定的部分",
        "其次明确指出证据不足的地方",
        "必要时向用户提出澄清性问题",
      ],

      style_instructions:
        "使用 persona_constraints 中定义的语气与表达习惯，但不得引入任何新事实。",
    };

    // --------------------------------------------------------
    // 4.3 AgentE：系统指令（写死，防跑偏）
    // --------------------------------------------------------

    const systemPrompt = `
你是 AgentE（最终人格回答者）。

【绝对规则】
1. 你只能使用 task_brief.grounded_facts 中提供的信息作为事实来源。
2. persona_constraints 仅用于风格、语气和决策方式，不得作为事实。
3. 如果证据不足，必须明确写出不确定性或向用户提问。
4. 禁止引入任何外部知识、常识或训练数据中的信息。

【输出格式】
你必须只输出 JSON，不得输出任何多余文本，格式如下：

{
  "text": "最终回答正文",
  "citations": [
    { "ref": "source_id 或 excerpt_id", "excerpt": "对应证据摘录" }
  ],
  "uncertainty": [
    "证据中缺失的信息点"
  ]
}
`;

    // --------------------------------------------------------
    // 4.4 调用 DeepSeek（真正执行 AgentE）
    // --------------------------------------------------------

    const completion = await openai.chat.completions.create({
      model: "deepseek-chat", // 你之后可以换成 deepseek-reasoner
      messages: [
        { role: "system", content: systemPrompt.trim() },
        {
          role: "user",
          content: JSON.stringify(task_brief),
        },
      ],
      temperature: 0.2, // 低随机性，防幻觉
    });

    const rawOutput = completion.choices[0].message.content || "";

    // --------------------------------------------------------
    // 4.5 解析并校验模型输出（AgentC 的最后一道防线）
    // --------------------------------------------------------

    // 清理 ```json ``` 包裹
    const cleaned = rawOutput
      .replace(/^```json/i, "")
      .replace(/^```/i, "")
      .replace(/```$/i, "")
      .trim();

    let parsed;
    try {
      parsed = JSON.parse(cleaned);
    } catch (err) {
      return res.status(502).json({
        error: "Model output is not valid JSON.",
        raw: rawOutput.slice(0, 1500),
      });
    }

    // 最小结构校验
    if (typeof parsed.text !== "string") {
      return res.status(502).json({
        error: "Invalid response schema: missing text.",
        parsed,
      });
    }

    if (!Array.isArray(parsed.citations)) parsed.citations = [];
    if (!Array.isArray(parsed.uncertainty)) parsed.uncertainty = [];

    // --------------------------------------------------------
    // 4.6 返回给前端
    // --------------------------------------------------------

    return res.json(parsed);
  } catch (err) {
    console.error("Server Error:", err);
    return res.status(500).json({
      error: "Internal server error",
      detail: String(err.message || err),
    });
  }
});

// ------------------------------------------------------------
// 5. 启动服务器
// ------------------------------------------------------------

app.listen(PORT, () => {
  console.log(`✅ AISELF backend running at http://localhost:${PORT}`);
});